package com.springboot.service;

import java.util.ArrayList;
import java.util.List;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.springboot.domain.Board;
import com.springboot.domain.BoardFormDto;
import com.springboot.repository.BoardRepository; // JpaRepository 대신 직접 정의한 인터페이스
import jakarta.transaction.Transactional; // Spring의 @Transactional 사용

import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl; // PageImpl 추가
import org.springframework.data.domain.PageRequest; // PageRequest 추가
import org.springframework.data.domain.Pageable; // Pageable 추가
import org.springframework.data.domain.Sort; // Sort 추가

@Service
public class BoardService {

    @Autowired
    private BoardRepository boardRepository;

    // 게시글 등록/수정하기
    @Transactional //
    public Long savePost (BoardFormDto boardDto) { //
        Board board = boardDto.toEntity(); //
        // JDBC에서는 ID가 없으면 insert, 있으면 update
        // createdDate, modifiedDate는 Repository에서 NOW()로 설정되므로 DTO에 직접 설정하지 않아도 됨.
        // 하지만 toEntity()에서 이 필드를 사용하므로, DTO에서 관리하는 것이 일관적일 수 있음.
        if (board.getId() == null || board.getId() == 0) {
             // 새 글 작성
        } else {
             // 수정
        }
        return boardRepository.save(board); //
    }

    // 전체 게시글 가져오기 (사용되지 않을 수 있음, 페이징된 목록이 주로 사용)
    @Transactional //
    public List<BoardFormDto> getBoardList() { //
        List<Board> boardList = boardRepository.findAll(); //
        List<BoardFormDto> boardDtoList = new ArrayList<>(); //
        for(Board board : boardList) { //
            BoardFormDto boardDto = BoardFormDto.builder() //
                    .id(board.getId()) //
                    .writerid(board.getWriterid()) //
                    .writer(board.getWriter()) //
                    .title(board.getTitle()) //
                    .content(board.getContent()) //
                    .createdDate(board.getCreatedDate()) //
                    .modifiedDate(board.getModifiedDate()) // 추가
                    .build(); //
            boardDtoList.add(boardDto); //
        }
        return boardDtoList; //
    }

    // 게시글 가져오기
    public Board getBoardById(Long id) { //
        Board board = boardRepository.findById(id); //
        // JPA의 .get() 대신 null 체크
        if (board == null) {
            throw new IllegalArgumentException("해당 게시글을 찾을 수 없습니다.");
        }
        return board; //
    }

    // 게시글 삭제하기
    public void deleteBoardById(Long id) { //
        boardRepository.deleteById(id); //
    }

    // 전체 게시글의 목록 개수, 정렬 가져오기 (JDBC에 맞게 수정)
    public Page<Board> listAll(int pageNum, String sortField, String sortDir) { //
        int pageSize = 5; //
        // JDBC에서는 PageRequest.of로 Sort 객체를 넘겨주기보다, Repository에서 직접 SQL 정렬 처리
        List<Board> content = boardRepository.findPage(pageNum, pageSize, sortField, sortDir);
        int totalItems = boardRepository.count();

        // PageImpl을 사용하여 Page 객체 생성
        Pageable pageable = PageRequest.of(pageNum - 1, pageSize, sortDir.equals("asc") ? Sort.by(sortField).ascending() : Sort.by(sortField).descending());
        return new PageImpl<>(content, pageable, totalItems);
    }
}