package com.springboot.repository;

import com.springboot.domain.Board;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Repository;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.time.LocalDateTime;
import java.util.List;

@Repository
public class BoardRepositoryImpl implements BoardRepository {

    private final JdbcTemplate jdbcTemplate;

    @Autowired
    public BoardRepositoryImpl(JdbcTemplate jdbcTemplate) {
        this.jdbcTemplate = jdbcTemplate;
    }

    // RowMapper 정의 (DB 컬럼 이름과 Board 필드 매핑)
    private RowMapper<Board> boardRowMapper = new RowMapper<Board>() {
        @Override
        public Board mapRow(ResultSet rs, int rowNum) throws SQLException {
            Board board = new Board();
            board.setId(rs.getLong("id"));
            board.setWriterid(rs.getString("writerid"));
            board.setWriter(rs.getString("writer"));
            board.setTitle(rs.getString("title"));
            board.setContent(rs.getString("content"));
            // createdDate, modifiedDate는 DB에 TIMESTAMP로 저장되어 있다고 가정
            board.setCreatedDate(rs.getTimestamp("createdDate") != null ? rs.getTimestamp("createdDate").toLocalDateTime() : null);
            board.setModifiedDate(rs.getTimestamp("modifiedDate") != null ? rs.getTimestamp("modifiedDate").toLocalDateTime() : null);
            return board;
        }
    };

    @Override
    public Long save(Board board) {
        if (board.getId() == null || board.getId() == 0) { // 새 게시글 (ID가 없거나 0이면 새 글)
            String sql = "INSERT INTO board (writerid, writer, title, content, createdDate, modifiedDate) VALUES (?, ?, ?, ?, NOW(), NOW())";
            jdbcTemplate.update(sql, board.getWriterid(), board.getWriter(), board.getTitle(), board.getContent());
            // JDBC는 save 후 자동 생성 ID 반환이 JpaRepository처럼 간단하지 않으므로,
            // 필요시 PreparedStatementCreatorFactory와 KeyHolder를 사용해야 합니다.
            // 여기서는 임시로 0L을 반환하지만, 실제 ID를 얻으려면 추가 로직 필요.
            // 또는 savePost 로직에서 ID가 필요 없다면 Long 대신 void로 변경.
            return 0L;
        } else { // 게시글 수정
            String sql = "UPDATE board SET writerid = ?, writer = ?, title = ?, content = ?, modifiedDate = NOW() WHERE id = ?";
            jdbcTemplate.update(sql, board.getWriterid(), board.getWriter(), board.getTitle(), board.getContent(), board.getId());
            return board.getId();
        }
    }

    @Override
    public Board findById(Long id) {
        String sql = "SELECT id, writerid, writer, title, content, createdDate, modifiedDate FROM board WHERE id = ?";
        try {
            return jdbcTemplate.queryForObject(sql, boardRowMapper, id);
        } catch (org.springframework.dao.EmptyResultDataAccessException e) {
            return null; // 찾는 게시글이 없으면 null 반환
        }
    }

    @Override
    public List<Board> findAll() {
        String sql = "SELECT id, writerid, writer, title, content, createdDate, modifiedDate FROM board ORDER BY createdDate DESC";
        return jdbcTemplate.query(sql, boardRowMapper);
    }

    @Override
    public void deleteById(Long id) {
        String sql = "DELETE FROM board WHERE id = ?";
        jdbcTemplate.update(sql, id);
    }

    @Override
    public int count() {
        String sql = "SELECT COUNT(*) FROM board";
        Integer count = jdbcTemplate.queryForObject(sql, Integer.class);
        return count != null ? count : 0;
    }

    @Override
    public List<Board> findPage(int pageNum, int pageSize, String sortField, String sortDir) {
        // ORDER BY 절 구성
        String orderBy = "";
        if (sortField != null && !sortField.isEmpty()) {
            orderBy = " ORDER BY " + sortField + " " + sortDir;
        } else {
            orderBy = " ORDER BY id DESC"; // 기본 정렬
        }

        // LIMIT 및 OFFSET 구성 (MySQL 기준)
        String sql = "SELECT id, writerid, writer, title, content, createdDate, modifiedDate FROM board" + orderBy + " LIMIT ? OFFSET ?";
        int offset = (pageNum - 1) * pageSize;
        return jdbcTemplate.query(sql, boardRowMapper, pageSize, offset);
    }
}