package com.springboot.controller;

import java.util.ArrayList;
import java.util.List;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import com.springboot.domain.Book;
import com.springboot.domain.Cart;
import com.springboot.domain.CartItem;
import com.springboot.domain.Customer;
import com.springboot.service.BookService;
import com.springboot.service.CartService;
import com.springboot.service.OrderService;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpSession;
import jakarta.validation.Valid;
import com.springboot.domain.Order;
import com.springboot.domain.OrderItem;
import com.springboot.domain.Shipping;
import com.springboot.service.OrderProService; // 추가된 JPA 서비스
import org.springframework.data.domain.Page; // 추가
import org.springframework.web.bind.annotation.RequestParam; // 추가
import org.springframework.web.servlet.ModelAndView; // 추가

@Controller
@RequestMapping(value = "/order")
public class OrderController {
    @Autowired
    private OrderService orderService; // 기존 In-memory OrderService
    @Autowired
    private CartService cartService;
    @Autowired
    private OrderProService orderProService; // 새로 추가된 Spring Data JPA 기반 서비스
    @Autowired
    private BookService bookService; // 추가

    Order order;
    List<Book> listofBooks;

    @GetMapping("/{cartId}")
    public String requestCartList(@PathVariable(value = "cartId") String cartId, Model model) { //
        Cart cart = cartService.read(cartId); // CartService.validateCart()는 정의되지 않음, read 사용
        order = new Order(); //
        listofBooks = new ArrayList<Book>(); //
        for(CartItem item : cart.getCartItems().values()) { //
            OrderItem orderItem = new OrderItem(); //
            Book book = item.getBook(); //
            listofBooks.add(book); //
            orderItem.setBookId(book.getBookId()); //
            orderItem.setQuantity(item.getQuantity()); //
            orderItem.setTotalPrice(item.getTotalPrice()); //
            order.getOrderItems().put(book.getBookId(), orderItem); //
        }
        order.setCustomer(new Customer()); //
        order.setShipping(new Shipping()); //
        order.setGrandTotal (cart.getGrandTotal()); //
        return "redirect:/order/orderCustomerInfo"; //
    }

    @GetMapping("/orderCustomerInfo")
    public String requestCustomerInfoForm(Model model){ //
        model.addAttribute("customer", order.getCustomer()); //
        return "orderCustomerInfo"; //
    }

    @PostMapping("/orderCustomerInfo")
    public String requestCustomerInfo(@ModelAttribute Customer customer, Model model){ //
        order.setCustomer(customer); //
        return "redirect:/order/orderShippingInfo"; //
    }

    @GetMapping("/orderShippingInfo")
    public String requestShippingInfoForm(Model model) { //
        model.addAttribute("shipping", order.getShipping()); //
        return "orderShippingInfo"; //
    }

    @PostMapping("/orderShippingInfo")
    public String requestShippingInfo(@Valid @ModelAttribute Shipping shipping, BindingResult bindingResult, Model model) { //
        if(bindingResult.hasErrors()) //
            return "orderShippingInfo"; //
        order.setShipping(shipping); //
        model.addAttribute("order", order); //
        return "redirect:/order/orderConfirmation"; //
    }

    @GetMapping("/orderConfirmation")
    public String requestConfirmation(Model model) { //
        model.addAttribute("bookList", listofBooks); //
        model.addAttribute("order", order); //
        return "orderConfirmation"; //
    }

    @PostMapping("/orderConfirmation") // 주문 완료 버튼 클릭 시
    public String requestConfirmationFinished (Model model) {
        model.addAttribute("order", order);
        orderProService.save(order); // 데이터베이스 테이블에 저장 메서드 호출
        return "redirect:/order/orderFinished";
    }

    @GetMapping("/orderFinished")
    public String requestFinished(HttpServletRequest request, Model model) { //
        // Long orderId = orderService.saveOrder (order); // in-memory orderRepository에 저장 (JPA 사용 시 불필요)
        // order.setOrderId(orderId); // 임시 orderId 설정 (JPA로 저장되었으므로 실제 사용은 안 함)
        model.addAttribute("order", order);
        HttpSession session = request.getSession(false);
        if(session != null) {
            session.invalidate();
        }
        return "orderFinished";
    }

    @GetMapping("/orderCancelled")
    public String requestCancelled(HttpServletRequest request) { //
        HttpSession session = request.getSession(false);
        if(session != null) {
            session.invalidate();
        }
        return "orderCancelled";
    }

    // 전체 주문 목록 가져오기 (관리자 모드)
    @GetMapping("/list")
    public String viewHomePage (Model model) {
        return viewPage(1, "orderId", "asc", model); // 기본 정렬 orderId 기준 오름차순
    }

    // 페이지별 주문 목록 보기 (관리자 모드)
    @GetMapping("/page")
    public String viewPage(@RequestParam("pageNum") int pageNum, @RequestParam("sortField") String sortField, @RequestParam("sortDir") String sortDir, Model model) { //
        Page<Order> page = orderProService.listAll(pageNum, sortField, sortDir); //
        List<Order> listOrders = page.getContent(); //

        model.addAttribute("currentPage", pageNum);
        model.addAttribute("totalPages", page.getTotalPages());
        model.addAttribute("totalItems", page.getTotalElements());
        model.addAttribute("sortField", sortField);
        model.addAttribute("sortDir", sortDir);
        model.addAttribute("reverseSortDir", sortDir.equals("asc") ? "desc" : "asc");
        model.addAttribute("orderList", listOrders);
        return "orderList";
    }

    // 주문 상세 보기 (관리자 모드)
    @GetMapping("/view/{id}")
    public ModelAndView viewOrder(@PathVariable(name = "id") Long id) { //
        Order order = orderProService.get(id); //
        List<Book> listofBooks = new ArrayList<Book>(); //
        for(OrderItem item : order.getOrderItems().values()) { //
            String bookId = item.getBookId(); //
            Book book = bookService.getBookById(bookId); //
            listofBooks.add(book); //
        }
        ModelAndView mav = new ModelAndView("orderView"); //
        mav.addObject("order", order); //
        mav.addObject("bookList", listofBooks); //
        return mav; //
    }

    // 주문 수정 페이지 표시 (관리자 모드)
    @GetMapping("/edit/{id}")
    public ModelAndView showEditOrder(@PathVariable(name = "id") Long id) { //
        Order order = orderProService.get(id); //
        List<Book> listofBooks = new ArrayList<Book>(); //
        for(OrderItem item : order.getOrderItems().values()) { //
            String bookId = item.getBookId(); //
            Book book = bookService.getBookById(bookId); //
            listofBooks.add(book); //
        }
        ModelAndView mav = new ModelAndView("orderEdit"); //
        mav.addObject("order", order); //
        mav.addObject("bookList", listofBooks); //
        return mav; //
    }

    // 주문 삭제 (관리자 모드)
    @GetMapping("/delete/{id}")
    public String deleteOrder(@PathVariable(name="id") Long id) { //
        orderProService.delete(id);
        return "redirect:/order/list";
    }

    // 전체 주문 삭제 (관리자 모드)
    @GetMapping("/deleteAll")
    public String deleteAllOrder() { //
        orderProService.deleteAll();
        return "redirect:/order/list";
    }

    // 주문 수정 저장 (관리자 모드)
    @PostMapping("/save")
    public String saveProduct (@ModelAttribute Order order) { //
        // 이전에 가져온 order 객체를 사용하여 엔티티를 병합 (Detach된 엔티티 문제 방지)
        Order save_order = orderProService.get(order.getOrderId()); //
        // 여기서는 Shipping 정보만 수정하는 것으로 가정. 실제로는 customer, orderItems도 고려
        save_order.setShipping(order.getShipping()); //
        orderProService.save(save_order); //
        return "redirect:/order/list"; //
    }
}