package com.springboot.controller;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import com.springboot.domain.Member;
import com.springboot.domain.MemberFormDto;
import com.springboot.service.MemberService;
import jakarta.validation.Valid;

@Controller
@RequestMapping(value = "/members")
public class MemberController {
    @Autowired
    MemberService memberService;
    @Autowired
    PasswordEncoder passwordEncoder;

    // 신규 회원 등록 페이지 출력하기
    @GetMapping(value ="/add")
    public String requestAddMemberForm (Model model) { //
        model.addAttribute("memberFormDto", new MemberFormDto()); //
        return "member/addMember"; //
    }

    // 신규 회원 등록하기
    @PostMapping(value ="/add")
    public String submitAddNewMember(@Valid MemberFormDto memberFormDto, BindingResult bindingResult, Model model) { //
        if(bindingResult.hasErrors()) { //
            return "member/addMember"; //
        }
        try { //
            Member member = Member.createMember (memberFormDto, passwordEncoder); //
            memberService.saveMember (member); //
        } catch (IllegalStateException e) { //
            model.addAttribute("errorMessage", e.getMessage()); //
            return "member/addMember"; //
        }
        return "redirect:/members"; //
    }

    // 회원 정보 수정 페이지 출력하기
    @GetMapping(value = "/update/{memberId}")
    public String requestUpdateMemberForm (@PathVariable(name = "memberId") String memberId, Model model) { //
        Member member = memberService.getMemberById(memberId); //
        // 비밀번호는 DTO에 담지 않고, 수정 시에만 다시 입력받도록 처리 (보안)
        MemberFormDto memberFormDto = new MemberFormDto();
        if (member != null) {
            memberFormDto.setMemberId(member.getMemberId());
            memberFormDto.setName(member.getName());
            memberFormDto.setPhone(member.getPhone());
            memberFormDto.setEmail(member.getEmail());
            memberFormDto.setAddress(member.getAddress());
        }
        model.addAttribute("memberFormDto", memberFormDto); //
        return "member/updateMember"; //
    }

    // 회원 정보 수정하기
    @PostMapping(value = "/update")
    public String submitUpdateMember (@Valid MemberFormDto memberFormDto, BindingResult bindingResult, Model model) { //
        if(bindingResult.hasErrors()) { //
            return "member/updateMember"; //
        }
        try { //
            Member existingMember = memberService.getMemberById(memberFormDto.getMemberId()); // 기존 회원 조회
            if (existingMember == null) {
                throw new IllegalStateException("해당 회원 ID를 찾을 수 없습니다.");
            }
            // 비밀번호가 입력된 경우에만 인코딩하여 업데이트
            if (memberFormDto.getPassword() != null && !memberFormDto.getPassword().isEmpty()) {
                existingMember.setPassword(passwordEncoder.encode(memberFormDto.getPassword()));
            }
            existingMember.setName(memberFormDto.getName());
            existingMember.setPhone(memberFormDto.getPhone());
            existingMember.setEmail(memberFormDto.getEmail());
            existingMember.setAddress(memberFormDto.getAddress());

            memberService.saveMember(existingMember); // saveMember가 merge 역할
        } catch (IllegalStateException e) { //
            model.addAttribute("errorMessage", e.getMessage()); //
            return "member/updateMember"; // 수정 페이지로 다시 돌아가도록 변경
        }
        return "redirect:/members"; //
    }

    // 회원 정보 삭제하기
    @GetMapping("/delete/{memberId}")
    public String deleteMember(@PathVariable(name = "memberId") String memberId) { //
        memberService.deleteMember (memberId); //
        return "redirect:/logout"; //
    }

    // 회원 가입 및 인증 시 인사말 페이지로 이동하기
    @GetMapping
    public String requestMain() { //
        return "redirect:/"; //
    }
}