package com.springboot.controller;

import java.util.List;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page; // Page 추가
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.ModelAttribute; // ModelAttribute 추가
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam; // RequestParam 추가
import com.springboot.domain.Board;
import com.springboot.domain.BoardFormDto;
import com.springboot.domain.Member; // Member 클래스 사용
import com.springboot.service.BoardService;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpSession;
import jakarta.validation.Valid;

@Controller
@RequestMapping(value = "/board")
public class BoardController {

    @Autowired
    private BoardService boardService;

    // 전체 게시글 목록 가져오기
    @GetMapping("/list")
    public String viewHomePage (Model model) { //
        return viewPage(1, "id", "desc", model); // 기본 정렬 id 기준 내림차순
    }

    // 전체 게시글 가져오기 (페이지네이션 및 정렬)
    @GetMapping("/page")
    public String viewPage (@RequestParam("pageNum") int pageNum, @RequestParam("sortField") String sortField, @RequestParam("sortDir") String sortDir, Model model) { //
        Page<Board> page = boardService.listAll(pageNum, sortField, sortDir); //
        List<Board> listBoard = page.getContent(); //

        model.addAttribute("currentPage", pageNum); //
        model.addAttribute("totalPages", page.getTotalPages()); //
        model.addAttribute("totalItems", page.getTotalElements()); //
        model.addAttribute("sortField", sortField); //
        model.addAttribute("sortDir", sortDir); //
        model.addAttribute("reverseSortDir", sortDir.equals("asc") ? "desc": "asc"); //
        model.addAttribute("boardList", listBoard); //
        return "board/list"; //
    }

    // 게시글 글쓰기 페이지 출력하기
    @GetMapping("/write")
    public String post(Model model, HttpSession session) { //
        // 로그인 정보를 가져와서 작성자 ID를 미리 채우거나 확인
        Member member = (Member) session.getAttribute("userLoginInfo");
        if (member == null) {
            // 로그인되어 있지 않으면 로그인 페이지로 리다이렉트 또는 에러 처리
            return "redirect:/login";
        }
        model.addAttribute("boardFormDto", new BoardFormDto());
        return "board/write"; //
    }

    // 게시글 글쓰기 저장하기
    @PostMapping("/write")
    public String write (@ModelAttribute BoardFormDto boardDto, HttpSession session) { //
        Member member = (Member) session.getAttribute("userLoginInfo");
        if (member == null) {
            return "redirect:/login"; // 로그인 정보 없으면 리다이렉트
        }
        boardDto.setWriterid(member.getMemberId()); // 세션에서 로그인 ID 설정
        boardDto.setWriter(member.getName()); // 세션에서 로그인 이름 설정

        boardService.savePost (boardDto); //
        return "redirect:/board/list"; //
    }

    // 게시글 상세 보기 (또는 수정 페이지 로드)
    @GetMapping("/view/{id}")
    public String requestUpdateMemberForm(@PathVariable(name = "id") Long id, HttpServletRequest httpServletRequest, Model model){ //
        Board board = boardService.getBoardById(id); //
        // Board Entity를 DTO로 변환하여 뷰에 전달
        BoardFormDto boardFormDto = BoardFormDto.builder()
                .id(board.getId())
                .writerid(board.getWriterid())
                .writer(board.getWriter())
                .title(board.getTitle())
                .content(board.getContent())
                .createdDate(board.getCreatedDate())
                .modifiedDate(board.getModifiedDate())
                .build();
        model.addAttribute("boardFormDto", boardFormDto); //

        HttpSession session = httpServletRequest.getSession(true); //
        Member member = (Member) session.getAttribute("userLoginInfo"); //

        model.addAttribute("buttonOk", false); // 수정/삭제 버튼 표시 여부
        if(member != null && board.getWriterid().equals(member.getMemberId())) { //
            model.addAttribute("buttonOk", true); //
        }
        return "board/view"; //
    }

    // 게시글 수정하기 (POST)
    @PostMapping("/update")
    public String submitUpdateMember (@Valid @ModelAttribute BoardFormDto boardDto, BindingResult bindingResult, Model model, HttpSession session) { //
        if(bindingResult.hasErrors()) { //
            // 에러 발생 시 작성자 ID, 이름 다시 DTO에 설정 (세션에서)
            Member member = (Member) session.getAttribute("userLoginInfo");
            if (member != null) {
                boardDto.setWriterid(member.getMemberId());
                boardDto.setWriter(member.getName());
            }
            model.addAttribute("buttonOk", true); // 수정 페이지이므로 버튼 계속 표시
            return "board/view"; //
        }

        try { //
            // 수정 시에도 작성자 ID와 이름을 세션에서 가져와 DTO에 설정 (보안 강화)
            Member member = (Member) session.getAttribute("userLoginInfo");
            if (member == null || !boardDto.getWriterid().equals(member.getMemberId())) {
                throw new IllegalStateException("수정 권한이 없습니다."); // 권한 없음 예외 처리
            }
            boardDto.setWriterid(member.getMemberId());
            boardDto.setWriter(member.getName());

            boardService.savePost (boardDto); //
        } catch (IllegalStateException e) { //
            model.addAttribute("errorMessage", e.getMessage()); //
            model.addAttribute("buttonOk", true); // 수정 페이지이므로 버튼 계속 표시
            return "board/view"; //
        }
        return "redirect:/board/list"; //
    }

    // 게시글 삭제하기
    @GetMapping("/delete/{id}")
    public String deleteOrder (@PathVariable(name = "id") Long id, HttpSession session) { //
        Board board = boardService.getBoardById(id);
        Member member = (Member) session.getAttribute("userLoginInfo");

        if (member == null || !board.getWriterid().equals(member.getMemberId())) {
            throw new IllegalStateException("삭제 권한이 없습니다."); // 권한 없음 예외 처리
        }
        boardService.deleteBoardById(id); //
        return "redirect:/board/list"; //
    }
}